'use client'

// React Imports
import { useState, useEffect, useCallback, useMemo } from 'react'

// Next Imports
import { useParams, useRouter } from 'next/navigation'

// MUI Imports
import useMediaQuery from '@mui/material/useMediaQuery'
import { styled, useTheme } from '@mui/material/styles'
import Typography from '@mui/material/Typography'
import IconButton from '@mui/material/IconButton'
import InputAdornment from '@mui/material/InputAdornment'
import Button from '@mui/material/Button'
import CircularProgress from '@mui/material/CircularProgress'
import Box from '@mui/material/Box'
import TextField from '@mui/material/TextField'

// Third-party Imports
import { signIn } from 'next-auth/react'
import { toast } from 'react-toastify'
import { Controller, useForm } from 'react-hook-form'
import { valibotResolver } from '@hookform/resolvers/valibot'
import { object, minLength, string, pipe, nonEmpty } from 'valibot'
import type { SubmitHandler } from 'react-hook-form'

// Type Imports
import type { SystemMode } from '@core/types'
import type { Locale } from '@/configs/i18n'

// Util Imports
import { getLocalizedUrl } from '@/utils/i18n'
import { getDictionaryClient } from '@/utils/getDictionaryClient'

// Hook Imports
import { useImageVariant } from '@core/hooks/useImageVariant'

// Styled Components
const StyledTextField = styled(TextField)({
  '& .MuiOutlinedInput-root': {
    backgroundColor: 'rgba(30, 41, 59, 0.8)',
    borderRadius: '8px',
    '& fieldset': {
      borderColor: '#0ea5e9',
      borderWidth: '2px'
    },
    '&:hover fieldset': {
      borderColor: '#38bdf8'
    },
    '&.Mui-focused fieldset': {
      borderColor: '#0ea5e9'
    }
  },
  '& .MuiInputBase-input': {
    color: '#fff',
    padding: '14px 16px',
    '&::placeholder': {
      color: 'rgba(255, 255, 255, 0.5)'
    }
  },
  '& .MuiInputAdornment-root': {
    color: 'rgba(255, 255, 255, 0.7)'
  },
  '& .MuiFormLabel-root': {
    color: 'rgba(255, 255, 255, 0.7)'
  },
  '& .MuiFormLabel-root.Mui-focused': {
    color: '#0ea5e9'
  }
})

type FormData = {
  identifier: string
  password: string
}

const Login = ({ mode }: { mode: SystemMode }) => {
  // States
  const [isPasswordShown, setIsPasswordShown] = useState(false)
  const [isLoading, setIsLoading] = useState(false)
  const [translations, setTranslations] = useState<Record<string, unknown>>({})

  // Translation function
  const t = useCallback(
    (key: string, fallback?: string) => {
      const keys = key.split('.')
      let value: unknown = translations

      for (const k of keys) {
        value = (value as Record<string, unknown>)?.[k]
      }

      return (value as string) || fallback || key
    },
    [translations]
  )

  // Vars
  const darkImg = '/images/pages/auth-mask-dark.png'
  const lightImg = '/images/pages/auth-mask-light.png'
  const darkIllustration = '/images/illustrations/auth/v2-login-dark.png'
  const lightIllustration = '/images/illustrations/auth/v2-login-light.png'
  const borderedDarkIllustration = '/images/illustrations/auth/v2-login-dark-border.png'
  const borderedLightIllustration = '/images/illustrations/auth/v2-login-light-border.png'

  // Hooks
  const router = useRouter()
  const { lang: locale } = useParams()
  const theme = useTheme()

  useMediaQuery(theme.breakpoints.down('md'))
  const authBackground = useImageVariant(mode, lightImg, darkImg)

  const isRTL = (locale as string) === 'ar'

  const characterIllustration = useImageVariant(
    mode,
    lightIllustration,
    darkIllustration,
    borderedLightIllustration,
    borderedDarkIllustration
  )

  const validationSchema = useMemo(
    () =>
      object({
        identifier: pipe(string(), nonEmpty(isRTL ? 'البريد الإلكتروني مطلوب' : 'Email is required')),
        password: pipe(
          string(),
          nonEmpty(isRTL ? 'كلمة المرور مطلوبة' : 'Password is required'),
          minLength(6, isRTL ? 'كلمة المرور يجب أن تكون 6 أحرف على الأقل' : 'Password must be at least 6 characters')
        )
      }),
    [isRTL]
  )

  const {
    control,
    handleSubmit,
    formState: { errors }
  } = useForm<FormData>({
    resolver: valibotResolver(validationSchema),
    defaultValues: {
      identifier: 'admin@example.com',
      password: 'password123'
    }
  })

  const handleClickShowPassword = () => setIsPasswordShown(show => !show)

  // Load translations
  useEffect(() => {
    const loadTranslations = async () => {
      try {
        const dict = await getDictionaryClient(locale as any)

        setTranslations(dict)
      } catch (error) {
        console.error('Failed to load translations:', error)
      }
    }

    loadTranslations()
  }, [locale])

  const onSubmit: SubmitHandler<FormData> = async (data: FormData) => {
    setIsLoading(true)

    try {
      const res = await signIn('credentials', {
        identifier: data.identifier,
        password: data.password,
        redirect: false
      })

      if (res && res.ok && res.error === null) {
        toast.success(t('auth.loginSuccess', 'تم تسجيل الدخول بنجاح!'), {
          position: 'top-right',
          autoClose: 1500,
          theme: 'colored'
        })

        setTimeout(() => {
          router.replace(getLocalizedUrl('/certificates', locale as Locale))
        }, 1500)
      } else {
        if (res?.error) {
          let errorMessage = t('auth.loginFailed', 'فشل تسجيل الدخول')

          try {
            const error = JSON.parse(res.error)

            if (error.messages) {
              errorMessage = error.messages
            } else if (error.message && Array.isArray(error.message)) {
              errorMessage = error.message[0]
            } else if (error.message) {
              errorMessage = error.message
            }
          } catch {
            errorMessage = res.error || errorMessage
          }

          toast.error(errorMessage, {
            position: 'top-right',
            autoClose: 5000,
            theme: 'colored'
          })
        }
      }
    } catch {
      toast.error(t('auth.unexpectedError', 'حدث خطأ غير متوقع'), {
        position: 'top-right',
        autoClose: 5000,
        theme: 'colored'
      })
    } finally {
      setIsLoading(false)
    }
  }

  return (
    <Box
      sx={{
        display: 'flex',
        maxHeight: '100vh',
        width: '100%',
        overflow: 'hidden'
      }}
    >
      {/* Left Section - Form */}
      <Box
        sx={{
          width: { xs: '100%', md: '45%' },
          minHeight: '100vh',
          background: 'linear-gradient(180deg, #1e293b 0%, #0f172a 100%)',
          display: 'flex',
          flexDirection: 'column',
          justifyContent: 'center',
          padding: { xs: 3, sm: 4, md: 6 },
          position: 'relative',
          direction: isRTL ? 'rtl' : 'ltr'
        }}
      >
        {/* Admin Title */}
        <Typography
          variant='h6'
          sx={{
            position: 'absolute',
            top: 24,
            ...(isRTL ? { right: 24 } : { left: 24 }),
            color: '#f59e0b',
            fontWeight: 700,
            display: 'flex',
            alignItems: 'center',
            gap: 1
          }}
        >
          🛡️ {t('auth.adminTitle', 'URS ADMIN')}
        </Typography>

        {/* Welcome Text */}
        <Box sx={{ mb: 4, textAlign: 'center' }}>
          <Typography
            variant='h4'
            sx={{
              color: '#f8fafc',
              fontWeight: 700,
              mb: 1,
              display: 'flex',
              alignItems: 'center',
              justifyContent: 'center',
              gap: 1
            }}
          >
            {t('auth.welcomeBack', 'Welcome back')} 👋
          </Typography>
          <Typography
            variant='body1'
            sx={{
              color: 'rgba(255, 255, 255, 0.6)',
              maxWidth: 400,
              mx: 'auto'
            }}
          >
            {t('auth.loginSubtitle', 'Sign in to manage certificates and continue operations')}
          </Typography>
        </Box>

        {/* Form */}
        <Box
          component='form'
          noValidate
          autoComplete='off'
          onSubmit={handleSubmit(onSubmit)}
          sx={{
            display: 'flex',
            flexDirection: 'column',
            gap: 3,
            maxWidth: 400,
            mx: 'auto',
            width: '100%'
          }}
        >
          {/* Email Field */}
          <Box>
            <Typography
              variant='body2'
              sx={{ color: 'rgba(255, 255, 255, 0.7)', mb: 1, textAlign: isRTL ? 'right' : 'left' }}
            >
              {t('auth.emailLabel', 'Email')}
            </Typography>
            <Controller
              name='identifier'
              control={control}
              rules={{ required: true }}
              render={({ field }) => (
                <StyledTextField
                  {...field}
                  autoFocus
                  fullWidth
                  placeholder={t('auth.emailPlaceholder', 'example@email.com')}
                  error={!!errors.identifier}
                  helperText={errors?.identifier?.message}
                  InputProps={{
                    endAdornment: (
                      <InputAdornment position='end'>
                        <i className='tabler-mail' style={{ color: 'rgba(255,255,255,0.5)' }} />
                      </InputAdornment>
                    )
                  }}
                  sx={{ direction: 'ltr' }}
                />
              )}
            />
          </Box>

          {/* Password Field */}
          <Box>
            <Typography
              variant='body2'
              sx={{ color: 'rgba(255, 255, 255, 0.7)', mb: 1, textAlign: isRTL ? 'right' : 'left' }}
            >
              {t('auth.passwordLabel', 'Password')}
            </Typography>
            <Controller
              name='password'
              control={control}
              rules={{ required: true }}
              render={({ field }) => (
                <StyledTextField
                  {...field}
                  fullWidth
                  placeholder={t('auth.passwordPlaceholder', '••••••••')}
                  type={isPasswordShown ? 'text' : 'password'}
                  error={!!errors.password}
                  helperText={errors?.password?.message}
                  InputProps={{
                    endAdornment: (
                      <InputAdornment position='end'>
                        <IconButton
                          edge='end'
                          onClick={handleClickShowPassword}
                          onMouseDown={e => e.preventDefault()}
                          sx={{ color: 'rgba(255,255,255,0.5)' }}
                        >
                          <i className={isPasswordShown ? 'tabler-eye' : 'tabler-eye-off'} />
                        </IconButton>
                      </InputAdornment>
                    )
                  }}
                  sx={{ direction: 'ltr' }}
                />
              )}
            />
          </Box>

          {/* Login Button */}
          <Button
            fullWidth
            variant='contained'
            type='submit'
            disabled={isLoading}
            sx={{
              mt: 2,
              py: 1.5,
              fontSize: '1rem',
              fontWeight: 600,
              borderRadius: 2,
              background: 'linear-gradient(90deg, #f59e0b 0%, #ea580c 100%)',
              '&:hover': {
                background: 'linear-gradient(90deg, #d97706 0%, #c2410c 100%)'
              },
              '&:disabled': {
                opacity: 0.7
              }
            }}
          >
            {isLoading ? (
              <Box sx={{ display: 'flex', alignItems: 'center', gap: 2 }}>
                <CircularProgress size={20} color='inherit' />
                {t('auth.loggingIn', 'Signing in...')}
              </Box>
            ) : (
              t('auth.login', 'Login')
            )}
          </Button>
        </Box>
      </Box>

      {/* Right Section - Image */}
      <Box
        sx={{
          width: '55%',
          display: { xs: 'none', md: 'flex' },
          alignItems: 'center',
          justifyContent: 'center',
          position: 'relative',
          overflow: 'hidden',
          background: 'linear-gradient(135deg, #0f172a 0%, #1e293b 100%)'
        }}
      >
        {/* Main Illustration */}
        <Box
          component='img'
          src={characterIllustration}
          alt='illustration'
          sx={{
            maxHeight: '100%',
            objectFit: 'contain',
            zIndex: 1
          }}
        />
      </Box>
    </Box>
  )
}

export default Login
