import { CustomUseMutationOptions } from '@/types'
import { MutationFunction, MutationKey, useMutation, useQueryClient } from '@tanstack/react-query'
import { AxiosError } from 'axios'
import { useRef } from 'react'
import { toast, type Id } from 'react-toastify'

export const useCustomMutation = <TVariable = any, TData = any, TError = TData, TContext = unknown>(
  mutationKey: MutationKey,
  mutationFn: MutationFunction<TData, TVariable>,
  options?: CustomUseMutationOptions<TData, AxiosError<TError>, TVariable, TContext>
) => {
  const queryClient = useQueryClient()
  const toastId = useRef<Id | null>(null)

  const notify = () => {
    toastId.current = toast.loading('تتم معالجة الطلب', { autoClose: false })
  }

  const update = (type: 'success' | 'error') => {
    let message = type === 'success' ? options?.promise?.message?.onSuccess : options?.promise?.message?.onError

    if (toastId.current !== null) {
      toast.update(toastId.current, {
        render: message,
        type,
        autoClose: 3000,
        isLoading: false
      })
    } else if (message) {
      type === 'success' ? toast.success(message) : toast.error(message)
    }
  }

  // Accept/forward ANY arg shapes (v5 3-arg or custom 4-arg)
  const onMutate = async (...args: any[]) => {
    if (!options?.withoutToastMessage && options?.toastMessageType === 'promise') {
      notify()
    }

    if (typeof options?.onMutate === 'function') {
      // forward exactly what TanStack gave us
      return await (options.onMutate as any)(...args)
    }
    // no custom context
    return undefined as TContext | undefined
  }

  const onSuccess = async (...args: any[]) => {
    // v5: [data, variables, context]
    // custom build: [data, variables, onMutateResult, context]
    const data = args[0] as TData

    try {
      if (typeof options?.onSuccess === 'function') {
        // forward exactly what TanStack gave us
        await (options.onSuccess as any)(...args)
      }

      if (options?.invalidateQueries) {
        await queryClient.invalidateQueries({
          queryKey: options.invalidateQueries
        })
      }

      if (options?.refetchQueries) {
        options.refetchQueries.forEach(rQ => queryClient.refetchQueries(rQ.filters, rQ.options))
      }

      if (!options?.withoutToastMessage) {
        if (options?.toastMessageType === 'promise') {
          update('success')
        } else {
          const messages: string[] = (data as any)?.messages
          toast.success(messages)
          if (messages?.length > 0) {
            messages?.map(message => toast.success(message))
          } else toast.success(messages)
        }
      }
    } catch {
      return
    }
  }

  const onError = async (...args: any[]) => {
    // v5: [error, variables, context]
    // custom build: [error, variables, onMutateResult, context]
    const error = args[0] as AxiosError<TError>

    try {
      if (!options?.withoutToastMessage) {
        if (options?.toastMessageType === 'promise') {
          update('error')
        } else if ((error as any)?.response?.data?.data?.email) {
          const msg: string = (error as any).response.data.data.email[0]
          toast.error(msg)
        } else if (Array.isArray((error as any)?.response?.data?.data)) {
          ;(error as any).response.data.data.forEach((m: any) => toast.error(String(m)))
        } else if ((error as any)?.response?.data?.message) {
          toast.error(String((error as any).response.data.message))
        } else if ((error as any)?.response?.data?.messages) {
          toast.error(String((error as any).response.data.messages))
        }
      }

      if (typeof options?.onError === 'function') {
        // forward exactly what TanStack gave us
        await (options.onError as any)(...args)
      }
    } catch {
      return
    }
  }

  return useMutation<TData, AxiosError<TError>, TVariable, TContext>({
    mutationKey,
    mutationFn,
    ...options,
    onMutate,
    onSuccess,
    onError
  })
}
