'use client'

import { useEffect, useMemo, useState } from 'react'
import { useParams } from 'next/navigation'

import type { Locale } from '@/configs/i18n'

type Dictionary = Record<string, any>

export type TranslationProps = {
  text: string
  dictionary?: Dictionary
  lang?: Locale
  fallback?: string
}

const dictionaries: Record<Locale, () => Promise<Dictionary>> = {
  en: () => import('@/data/dictionaries/en.json').then(module => module.default),
  ar: () => import('@/data/dictionaries/ar.json').then(module => module.default)
}

const getValue = (dictionary: Dictionary | undefined, key: string): string | undefined => {
  if (!dictionary) return undefined

  return key.split('.').reduce((acc: any, part) => (acc && part in acc ? acc[part] : undefined), dictionary)
}

const Translation = ({ text, dictionary, lang, fallback }: TranslationProps) => {
  const params = useParams()
  const resolvedLang = lang ?? (typeof params.lang === 'string' ? (params.lang as Locale) : 'en')
  const [loadedDictionary, setLoadedDictionary] = useState<Dictionary | undefined>(dictionary)

  useEffect(() => {
    if (dictionary) {
      setLoadedDictionary(dictionary)
      return
    }

    dictionaries[resolvedLang]()
      .then(setLoadedDictionary)
      .catch(() => setLoadedDictionary(undefined))
  }, [dictionary, resolvedLang])

  const translated = useMemo(
    () => getValue(loadedDictionary, text) ?? fallback ?? text,
    [loadedDictionary, text, fallback]
  )

  return <>{translated}</>
}

export default Translation
